# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- AssembleFeatures
#'
#'     Assembles feature columns into a vector column of features.
#' @param allowImages Allow featurization of images
#' @param columnsToFeaturize Columns to featurize
#' @param featuresCol The name of the features column
#' @param numberOfFeatures Number of features to hash string columns to
#' @param oneHotEncodeCategoricals One-hot encode categoricals
#' @export
ml_assemble_features <- function(x, allowImages=FALSE, columnsToFeaturize=NULL, featuresCol="features", numberOfFeatures=NULL, oneHotEncodeCategoricals=TRUE, only.model=FALSE)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.AssembleFeatures"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setAllowImages", as.logical(allowImages)) %>%
    invoke("setColumnsToFeaturize", as.array(columnsToFeaturize)) %>%
    invoke("setFeaturesCol", featuresCol) %>%
    invoke("setNumberOfFeatures", as.integer(numberOfFeatures)) %>%
    invoke("setOneHotEncodeCategoricals", as.logical(oneHotEncodeCategoricals))
  mod_model_raw <- mod_parameterized %>%
    invoke("fit", df)

  mod_model <- sparklyr:::new_ml_model(mod_parameterized, mod_model_raw, mod_model_raw)

  if (only.model)
    return(mod_model)

  transformed <- invoke(mod_model$model, "transform", df)

  sdf_register(transformed)
}
