# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- Featurize
#'
#'     Featurizes a dataset.  Converts the specified columns to feature columns.
#' @param allowImages Allow featurization of images
#' @param featureColumns Feature columns
#' @param numberOfFeatures Number of features to hash string columns to
#' @param oneHotEncodeCategoricals One-hot encode categoricals
#' @export
ml_featurize <- function(x, allowImages=FALSE, featureColumns=NULL, numberOfFeatures=262144, oneHotEncodeCategoricals=TRUE, only.model=FALSE)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.Featurize"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setAllowImages", as.logical(allowImages)) %>%
    invoke("setFeatureColumns", featureColumns) %>%
    invoke("setNumberOfFeatures", as.integer(numberOfFeatures)) %>%
    invoke("setOneHotEncodeCategoricals", as.logical(oneHotEncodeCategoricals))
  mod_model_raw <- mod_parameterized %>%
    invoke("fit", df)

  mod_model <- sparklyr:::new_ml_model(mod_parameterized, mod_model_raw, mod_model_raw)

  if (only.model)
    return(mod_model)

  transformed <- invoke(mod_model$model, "transform", df)

  sdf_register(transformed)
}
