# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- LightGBMRegressor
#'
#' 
#' @param alpha parameter for Huber loss and Quantile regression
#' @param application Regression application, regression_l2, regression_l1, huber, fair, poisson, quantile, mape, gamma or tweedie
#' @param baggingFraction Bagging fraction
#' @param baggingFreq Bagging frequence
#' @param baggingSeed Bagging seed
#' @param defaultListenPort The default listen port on executors, used for testing
#' @param featureFraction Feature fraction
#' @param featuresCol features column name
#' @param labelCol label column name
#' @param learningRate Learning rate or shrinkage rate
#' @param maxBin Max bin
#' @param maxDepth Max depth
#' @param minSumHessianInLeaf Minimal sum hessian in one leaf
#' @param numIterations Number of iterations, LightGBM constructs num_class * num_iterations trees
#' @param numLeaves Number of leaves
#' @param parallelism Tree learner parallelism, can be set to data_parallel or voting_parallel
#' @param predictionCol prediction column name
#' @param timeout Timeout in seconds
#' @export
ml_light_gbmregressor <- function(x, alpha=0.9, application="regression", baggingFraction=1.0, baggingFreq=0, baggingSeed=3, defaultListenPort=12400, featureFraction=1.0, featuresCol="features", labelCol="label", learningRate=0.1, maxBin=255, maxDepth=-1, minSumHessianInLeaf=0.001, numIterations=100, numLeaves=31, parallelism="data_parallel", predictionCol="prediction", timeout=120.0, only.model=FALSE)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.LightGBMRegressor"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setAlpha", as.double(alpha)) %>%
    invoke("setApplication", application) %>%
    invoke("setBaggingFraction", as.double(baggingFraction)) %>%
    invoke("setBaggingFreq", as.integer(baggingFreq)) %>%
    invoke("setBaggingSeed", as.integer(baggingSeed)) %>%
    invoke("setDefaultListenPort", as.integer(defaultListenPort)) %>%
    invoke("setFeatureFraction", as.double(featureFraction)) %>%
    invoke("setFeaturesCol", featuresCol) %>%
    invoke("setLabelCol", labelCol) %>%
    invoke("setLearningRate", as.double(learningRate)) %>%
    invoke("setMaxBin", as.integer(maxBin)) %>%
    invoke("setMaxDepth", as.integer(maxDepth)) %>%
    invoke("setMinSumHessianInLeaf", as.double(minSumHessianInLeaf)) %>%
    invoke("setNumIterations", as.integer(numIterations)) %>%
    invoke("setNumLeaves", as.integer(numLeaves)) %>%
    invoke("setParallelism", parallelism) %>%
    invoke("setPredictionCol", predictionCol) %>%
    invoke("setTimeout", as.double(timeout))
  mod_model_raw <- mod_parameterized %>%
    invoke("fit", df)

  mod_model <- sparklyr:::new_ml_model(mod_parameterized, mod_model_raw, mod_model_raw)

  if (only.model)
    return(mod_model)

  transformed <- invoke(mod_model$model, "transform", df)

  sdf_register(transformed)
}
