# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- MultiColumnAdapter
#'
#'     Takes a unary pipeline stage and a list of input/output column pairs and
#'         applies the pipeline stage to each column after a fit
#' @param baseStage base pipeline stage to apply to every column
#' @param inputCols list of column names encoded as a string
#' @param outputCols list of column names encoded as a string
#' @export
ml_multi_column_adapter <- function(x, baseStage=NULL, inputCols=NULL, outputCols=NULL, only.model=FALSE)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.MultiColumnAdapter"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setBaseStage", baseStage) %>%
    invoke("setInputCols", as.array(inputCols)) %>%
    invoke("setOutputCols", as.array(outputCols))
  mod_model_raw <- mod_parameterized %>%
    invoke("fit", df)

  mod_model <- sparklyr:::new_ml_model(mod_parameterized, mod_model_raw, mod_model_raw)

  if (only.model)
    return(mod_model)

  transformed <- invoke(mod_model$model, "transform", df)

  sdf_register(transformed)
}
