# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- TrainRegressor
#'
#'     Use ``TrainRegressor`` to train a regression model on a dataset.
#'     
#'         Below is an example that uses ``TrainRegressor``.  Given a DataFrame,
#'         myDataFrame, with a label column, "MyLabel", split the DataFrame into
#'         train and test sets.  Train a regressor on the dataset with a solver,
#'         such as l-bfgs:
#'     
#'         >>> from mmlspark.TrainRegressor import TrainRegressor
#'         >>> from pysppark.ml.regression import LinearRegression
#'         >>> lr = LinearRegression().setSolver("l-bfgs").setRegParam(0.1).setElasticNetParam(0.3)
#'         >>> model = TrainRegressor(model=lr, labelCol="MyLabel", numFeatures=1 << 18).fit(train)
#'     
#'         Now that you have a model, you can score the regressor on the test data:
#'     
#'         >>> scoredData = model.transform(test)
#' @param labelCol The name of the label column
#' @param model Regressor to run
#' @param numFeatures Number of features to hash to
#' @export
ml_train_regressor <- function(x, labelCol=NULL, model=NULL, numFeatures=0, only.model=FALSE)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.TrainRegressor"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setLabelCol", labelCol) %>%
    invoke("setModel", model) %>%
    invoke("setNumFeatures", as.integer(numFeatures))
  mod_model_raw <- mod_parameterized %>%
    invoke("fit", df)

  mod_model <- sparklyr:::new_ml_model(mod_parameterized, mod_model_raw, mod_model_raw)

  if (only.model)
    return(mod_model)

  transformed <- invoke(mod_model$model, "transform", df)

  sdf_register(transformed)
}
