# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- ImageFeaturizer
#'
#'     The ``ImageFeaturizer`` relies on a ``CNTKModel`` to do the
#'         featurization of the image(s).  One can set this model using the
#'         modelLocation parameter.  To map the nodes of the ``CNTKModel`` onto the
#'         standard "layers" structure of a feed forward neural net, one needs to
#'         supply a list of node names that range from the output node, back
#'         towards the input node of the CNTK Function.  This list does not need to
#'         be exhaustive, and is provided to you if you use a model downloaded
#'         fromt ``ModelDownloader``.  One can find this layer list in the schema
#'         of the downloaded model.
#'     
#'         The ``ImageFeaturizer`` takes an input column of images (the type
#'         returned by the ``ImageReader``), automatically resizes them to fit the
#'         ``CNTKModel``'s inputs, and feeds them through a pre-trained CNTK model.
#'         One can truncate the model using the ``cutOutputLayers`` parameter that
#'         determines how many layers to truncate from the output of the network.
#'         For example, layer=0 means that no layers are removed, layer=2 means
#'         that the ``ImageFeaturizer`` returns the activations of the layer that
#'         is two layers from the output layer, and so on.
#' @param cntkModel The internal CNTK model used in the featurizer
#' @param cutOutputLayers The number of layers to cut off the end of the network, 0 leaves the network intact, 1 removes the output layer, etc
#' @param dropNa Whether to drop na values before mapping
#' @param inputCol The name of the input column
#' @param layerNames Array with valid CNTK nodes to choose from, the first entries of this array should be closer to the output node
#' @param outputCol The name of the output column
#' @export
ml_image_featurizer <- function(x, cntkModel=NULL, cutOutputLayers=1, dropNa=TRUE, inputCol=NULL, layerNames=NULL, outputCol=NULL)
{
  if (unfit.model) {
    sc <- x
  } else {
    df <- spark_dataframe(x)
    sc <- spark_connection(df)
  }
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.ImageFeaturizer"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setCntkModel", cntkModel) %>%
    invoke("setCutOutputLayers", as.integer(cutOutputLayers)) %>%
    invoke("setDropNa", as.logical(dropNa)) %>%
    invoke("setInputCol", inputCol) %>%
    invoke("setLayerNames", as.array(layerNames)) %>%
    invoke("setOutputCol", outputCol)

  transformed <- invoke(mod_parameterized, "transform", df)

  sdf_register(transformed)
}
