# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- ImageLIME
#'
#' 
#' @param cellSize Number that controls the size of the superpixels
#' @param inputCol The name of the input column
#' @param labelCol The name of the label column
#' @param localModelPartitions The number of partitions to coalesce to to fit the local model
#' @param model Model to try to locally approximate
#' @param modelPartitions The number of partitions to create for evaluating the model
#' @param modifier Controls the trade-off spatial and color distance
#' @param nSamples The number of samples to generate
#' @param outputCol The name of the output column
#' @param samplingFraction The fraction of superpixels to keep on
#' @param superpixelCol The column holding the superpixel decompositions
#' @export
ml_image_lime <- function(x, cellSize=16.0, inputCol=NULL, labelCol=NULL, localModelPartitions=NULL, model=NULL, modelPartitions=NULL, modifier=130.0, nSamples=900, outputCol=NULL, samplingFraction=0.3, superpixelCol="superpixels")
{
  if (unfit.model) {
    sc <- x
  } else {
    df <- spark_dataframe(x)
    sc <- spark_connection(df)
  }
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.ImageLIME"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setCellSize", as.double(cellSize)) %>%
    invoke("setInputCol", inputCol) %>%
    invoke("setLabelCol", labelCol) %>%
    invoke("setLocalModelPartitions", as.integer(localModelPartitions)) %>%
    invoke("setModel", model) %>%
    invoke("setModelPartitions", as.integer(modelPartitions)) %>%
    invoke("setModifier", as.double(modifier)) %>%
    invoke("setNSamples", as.integer(nSamples)) %>%
    invoke("setOutputCol", outputCol) %>%
    invoke("setSamplingFraction", as.double(samplingFraction)) %>%
    invoke("setSuperpixelCol", superpixelCol)

  transformed <- invoke(mod_parameterized, "transform", df)

  sdf_register(transformed)
}
