# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- FindSimilarFace
#'
#' 
#' @param concurrency max number of concurrent calls
#' @param concurrentTimeout max number seconds to wait on futures if concurrency >= 1
#' @param errorCol column to hold http errors
#' @param faceId faceId of the query face. User needs to call FaceDetect first to get a valid faceId. Note that this faceId is not persisted and will expire 24 hours after the detection call.
#' @param faceIds  An array of candidate faceIds. All of them are created by FaceDetect and the faceIds will expire 24 hours after the detection call. The number of faceIds is limited to 1000. Parameter faceListId, largeFaceListId and faceIds should not be provided at the same time.
#' @param faceListId  An existing user-specified unique candidate face list, created in FaceList - Create. Face list contains a set of persistedFaceIds which are persisted and will never expire. Parameter faceListId, largeFaceListId and faceIds should not be provided at the same time.
#' @param handler Which strategy to use when handling requests
#' @param largeFaceListId  An existing user-specified unique candidate large face list, created in LargeFaceList - Create. Large face list contains a set of persistedFaceIds which are persisted and will never expire. Parameter faceListId, largeFaceListId and faceIds should not be provided at the same time.
#' @param maxNumOfCandidatesReturned  Optional parameter. The number of top similar faces returned. The valid range is [1, 1000].It defaults to 20.
#' @param mode  Optional parameter. Similar face searching mode. It can be 'matchPerson' or 'matchFace'. It defaults to 'matchPerson'.
#' @param outputCol The name of the output column
#' @param subscriptionKey the API key to use
#' @param url Url of the service
#' @export
ml_find_similar_face <- function(x, concurrency=1, concurrentTimeout=100.0, errorCol=NULL, faceId=NULL, faceIds=NULL, faceListId=NULL, handler="UserDefinedFunction(<function2>,StringType,None)", largeFaceListId=NULL, maxNumOfCandidatesReturned=NULL, mode=NULL, outputCol=NULL, subscriptionKey=NULL, url=NULL)
{
  if (unfit.model) {
    sc <- x
  } else {
    df <- spark_dataframe(x)
    sc <- spark_connection(df)
  }
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.FindSimilarFace"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setConcurrency", as.integer(concurrency)) %>%
    invoke("setConcurrentTimeout", as.double(concurrentTimeout)) %>%
    invoke("setErrorCol", errorCol) %>%
    invoke("setFaceId", faceId) %>%
    invoke("setFaceIds", faceIds) %>%
    invoke("setFaceListId", faceListId) %>%
    invoke("setHandler", handler) %>%
    invoke("setLargeFaceListId", largeFaceListId) %>%
    invoke("setMaxNumOfCandidatesReturned", maxNumOfCandidatesReturned) %>%
    invoke("setMode", mode) %>%
    invoke("setOutputCol", outputCol) %>%
    invoke("setSubscriptionKey", subscriptionKey) %>%
    invoke("setUrl", url)

  transformed <- invoke(mod_parameterized, "transform", df)

  sdf_register(transformed)
}
