# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- PartitionSample
#'
#'     Sampling mode.  The options are:
#'     
#'             - AssignToPartition
#'             - RandomSample
#'             - Head
#'     
#'         The default is RandomSample.
#'     
#'         Relevant parameters for the different modes are:
#'     
#'         - When the mode is AssignToPartition:
#'             - seed - the seed for random partition assignment.
#'             - numParts - the number of partitions.  The Default is 10.
#'             - newColName - the name of the partition column.
#'               The default is "Partition".
#'         - When the mode is RandomSample:
#'             - mode - Absolute or Percentage
#'             - count - the number of rows to assign to each partition when
#'               Absolute
#'             - percent - the percentage per partition when Percentage
#'         - When the mode is Head:
#'             - count - the number of rows
#' @param count Number of rows to return
#' @param mode AssignToPartition, RandomSample, or Head
#' @param newColName Name of the partition column
#' @param numParts Number of partitions
#' @param percent Percent of rows to return
#' @param rsMode Absolute or Percentage
#' @param seed Seed for random operations
#' @export
ml_partition_sample <- function(x, count=1000, mode="RandomSample", newColName="Partition", numParts=10, percent=0.01, rsMode="Percentage", seed=-1)
{
  if (unfit.model) {
    sc <- x
  } else {
    df <- spark_dataframe(x)
    sc <- spark_connection(df)
  }
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.PartitionSample"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setCount", as.integer(count)) %>%
    invoke("setMode", mode) %>%
    invoke("setNewColName", newColName) %>%
    invoke("setNumParts", as.integer(numParts)) %>%
    invoke("setPercent", as.double(percent)) %>%
    invoke("setRsMode", rsMode) %>%
    invoke("setSeed", as.integer(seed))

  transformed <- invoke(mod_parameterized, "transform", df)

  sdf_register(transformed)
}
