# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- ValueIndexer
#'
#'     Fits a dictionary of values from the input column.
#'     
#'         The ``ValueIndexer`` generates a model, then transforms a column to a
#'         categorical column of the given array of values.  It is similar to
#'         ``StringIndexer`` except that it can be used on any value types.
#' @param inputCol The name of the input column
#' @param outputCol The name of the output column
#' @export
ml_value_indexer <- function(x, inputCol=NULL, outputCol=NULL, unfit.model=FALSE, only.model=FALSE)
{
  if (unfit.model) {
    sc <- x
  } else {
    df <- spark_dataframe(x)
    sc <- spark_connection(df)
  }
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.ValueIndexer"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setInputCol", inputCol) %>%
    invoke("setOutputCol", outputCol)
  if (unfit.model)
    return(mod_parameterized)
  mod_model_raw <- mod_parameterized %>%
    invoke("fit", df)

  mod_model <- sparklyr:::new_ml_model(mod_parameterized, mod_model_raw, mod_model_raw)

  if (only.model)
    return(mod_model)

  transformed <- invoke(mod_model$model, "transform", df)

  sdf_register(transformed)
}
