# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- ImageLIME
#'
#' 
#' @param cellSize Number that controls the size of the superpixels
#' @param inputCol The name of the input column
#' @param model Model to try to locally approximate
#' @param modifier Controls the trade-off spatial and color distance
#' @param nSamples The number of samples to generate
#' @param outputCol The name of the output column
#' @param predictionCol prediction column name
#' @param regularization regularization param for the lasso
#' @param samplingFraction The fraction of superpixels to keep on
#' @param superpixelCol The column holding the superpixel decompositions
#' @export
ml_image_lime <- function(x, cellSize=16.0, inputCol=NULL, model=NULL, modifier=130.0, nSamples=900, outputCol=NULL, predictionCol="prediction", regularization=0.0, samplingFraction=0.3, superpixelCol="superpixels")
{
  if (unfit.model) {
    sc <- x
  } else {
    df <- spark_dataframe(x)
    sc <- spark_connection(df)
  }
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.ImageLIME"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setCellSize", as.double(cellSize)) %>%
    invoke("setInputCol", inputCol) %>%
    invoke("setModel", model) %>%
    invoke("setModifier", as.double(modifier)) %>%
    invoke("setNSamples", as.integer(nSamples)) %>%
    invoke("setOutputCol", outputCol) %>%
    invoke("setPredictionCol", predictionCol) %>%
    invoke("setRegularization", as.double(regularization)) %>%
    invoke("setSamplingFraction", as.double(samplingFraction)) %>%
    invoke("setSuperpixelCol", superpixelCol)

  transformed <- invoke(mod_parameterized, "transform", df)

  sdf_register(transformed)
}
