# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- TabularLIME
#'
#' 
#' @param inputCol The name of the input column
#' @param model Model to try to locally approximate
#' @param nSamples The number of samples to generate
#' @param outputCol The name of the output column
#' @param predictionCol prediction column name
#' @param regularization regularization param for the lasso
#' @param samplingFraction The fraction of superpixels to keep on
#' @export
ml_tabular_lime <- function(x, inputCol=NULL, model=NULL, nSamples=1000, outputCol=NULL, predictionCol="prediction", regularization=0.0, samplingFraction=0.3, unfit.model=FALSE, only.model=FALSE)
{
  if (unfit.model) {
    sc <- x
  } else {
    df <- spark_dataframe(x)
    sc <- spark_connection(df)
  }
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.TabularLIME"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setInputCol", inputCol) %>%
    invoke("setModel", model) %>%
    invoke("setNSamples", as.integer(nSamples)) %>%
    invoke("setOutputCol", outputCol) %>%
    invoke("setPredictionCol", predictionCol) %>%
    invoke("setRegularization", as.double(regularization)) %>%
    invoke("setSamplingFraction", as.double(samplingFraction))
  if (unfit.model)
    return(mod_parameterized)
  mod_model_raw <- mod_parameterized %>%
    invoke("fit", df)

  mod_model <- sparklyr:::new_ml_model(mod_parameterized, mod_model_raw, mod_model_raw)

  if (only.model)
    return(mod_model)

  transformed <- invoke(mod_model$model, "transform", df)

  sdf_register(transformed)
}
