# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- TuneHyperparameters
#'
#'     Tunes model hyperparameters
#'     
#'         Supports distributed hyperparameter tuning through randomized grid search.
#'         In the near future will support Nelder-Mead and kernel density optimization.
#' @param evaluationMetric Metric to evaluate models with
#' @param models Estimators to run
#' @param numFolds Number of folds
#' @param numRuns Termination criteria for randomized search
#' @param parallelism The number of models to run in parallel
#' @param paramSpace Parameter space for generating hyperparameters
#' @param seed Random number generator seed
#' @export
ml_tune_hyperparameters <- function(x, evaluationMetric=NULL, models=NULL, numFolds=NULL, numRuns=NULL, parallelism=NULL, paramSpace=NULL, seed=0, unfit.model=FALSE, only.model=FALSE)
{
  if (unfit.model) {
    sc <- x
  } else {
    df <- spark_dataframe(x)
    sc <- spark_connection(df)
  }
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.TuneHyperparameters"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setEvaluationMetric", evaluationMetric) %>%
    invoke("setModels", models) %>%
    invoke("setNumFolds", as.integer(numFolds)) %>%
    invoke("setNumRuns", as.integer(numRuns)) %>%
    invoke("setParallelism", as.integer(parallelism)) %>%
    invoke("setParamSpace", paramSpace) %>%
    invoke("setSeed", as.integer(seed))
  if (unfit.model)
    return(mod_parameterized)
  mod_model_raw <- mod_parameterized %>%
    invoke("fit", df)

  mod_model <- sparklyr:::new_ml_model(mod_parameterized, mod_model_raw, mod_model_raw)

  if (only.model)
    return(mod_model)

  transformed <- invoke(mod_model$model, "transform", df)

  sdf_register(transformed)
}
