# Copyright (C) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See LICENSE in project root for information.

#' Spark ML -- TabularLIMEModel
#'
#' 
#' @param columnMeans the means of each of the columns for perturbation
#' @param columnSTDs the standard deviations of each of the columns for perturbation
#' @param inputCol The name of the input column
#' @param model Model to try to locally approximate
#' @param nSamples The number of samples to generate
#' @param outputCol The name of the output column
#' @param predictionCol prediction column name
#' @param regularization regularization param for the lasso
#' @param samplingFraction The fraction of superpixels to keep on
#' @export
ml_tabular_limemodel <- function(x, columnMeans=NULL, columnSTDs=NULL, inputCol=NULL, model=NULL, nSamples=NULL, outputCol=NULL, predictionCol="prediction", regularization=NULL, samplingFraction=NULL)
{
  if (unfit.model) {
    sc <- x
  } else {
    df <- spark_dataframe(x)
    sc <- spark_connection(df)
  }
  env <- new.env(parent = emptyenv())

  env$model <- "com.microsoft.ml.spark.TabularLIMEModel"
  mod <- invoke_new(sc, env$model)

  mod_parameterized <- mod %>%
    invoke("setColumnMeans", columnMeans) %>%
    invoke("setColumnSTDs", columnSTDs) %>%
    invoke("setInputCol", inputCol) %>%
    invoke("setModel", model) %>%
    invoke("setNSamples", as.integer(nSamples)) %>%
    invoke("setOutputCol", outputCol) %>%
    invoke("setPredictionCol", predictionCol) %>%
    invoke("setRegularization", as.double(regularization)) %>%
    invoke("setSamplingFraction", as.double(samplingFraction))

  transformed <- invoke(mod_parameterized, "transform", df)

  sdf_register(transformed)
}
